% Loading External Image Maps Example
%
% This example demonstrates how to use an external image as the initial
% pressure distribution in the simulation and detection of photoacoustic
% waves within a two-dimensional homogeneous propagation medium. It builds
% on the Homogeneous Propagation Medium Example.  
%
% author: Bradley Treeby
% date: 30th June 2009
% last update: 16th July 2009
%  
% This example is part of the k-Wave Toolbox (http://www.k-wave.org)

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% load the initial pressure distribution from an image and scale the
% magnitude
p0_magnitude = 3;
p0 = p0_magnitude*loadImage('EXAMPLE_source_one.png');

% create the computational grid
Nx = 128;           % number of pixels in the x (column) direction
Nz = 128;           % number of pixels in the z (row) direction
dx = 0.1e-3;        % pixel width [m]
dz = 0.1e-3;        % pixel height [m]
kgrid = makeGrid(Nx, dx, Nz, dz);

% resize the input image to the desired number of pixels
p0 = resize(p0, Nx, Nz);

% define the properties of the propagation medium
c = 1500;           % [m/s]
rho = 1000;         % [kg/m^3]

% define a centered circular sensor
sensor_radius = 4e-3;   % [m]
num_sensor_points = 50;
sensor_mask = makeCartCircle(sensor_radius, num_sensor_points);

% run the simulation
sensor_data = kspaceFirstOrder2D(p0, kgrid, c, rho, 'auto', sensor_mask);

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the initial pressure and sensor distribution
figure;
imagesc(kgrid.x(1,:)*1e3, kgrid.z(:,1)*1e3, p0 + cart2grid(kgrid, sensor_mask), [-1 1]);
colormap(getColorMap);
ylabel('z-position [mm]');
xlabel('x-position [mm]');
axis image;

% plot the simulated sensor data
figure;
imagesc(sensor_data, [-1, 1]);
colormap(getColorMap);
ylabel('Sensor Position');
xlabel('Time Step');
colorbar;